<?php
/*
Plugin Name: Cryout Serious Image Sizes
Plugin URI: http://www.cryoutcreations.eu/
Description: This plugin displays and adds the ability to customize registered image sizes.
Author: Cryout Creations
Author URI: http://www.cryoutcreations.eu/
Version: 0.5
License: GPL v3 - http://www.gnu.org/licenses/gpl-3.0.html
*/

class Cryout_SeriousImageSizes {

	private $version = '0.5';
	private $the_page = NULL;
	private $page_url = 'options-general.php?page=cryout-seriousimagesettings';
	
	private $sizes = array(); // main image sizes array, populated on init
	
	private $known_theme = FALSE;
	private $known_themes = array( 'anima', 'bravada', 'fluida', 'kahuna', 'roseta', 'septera' );

	function __construct(){		
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'actions_links' ), -10 );
		add_filter( 'plugin_row_meta', array( $this, 'meta_links' ), 10, 2 );
		add_action( 'admin_menu', array( $this, 'register_menu_page' ) );
		
		// fetch and override registered image sizes array
		add_action( 'init', array( $this, 'parse_image_sizes'), 999 );
		
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_init', array( $this, 'save_settings' ) );
		
		// fetch options
		$this->options = $this->load_settings();

		add_action( 'admin_enqueue_scripts', array($this, 'styling') );
		//add_action( 'admin_enqueue_scripts', array($this, 'scripting') );
	} // __construct()
	
	/*** SETTINGS MANAGEMENT ***/

	/**
	 * Register the plugin's settings
	 */
	public function register_settings() {
		register_setting( 'cryout_sis_group', 'cryout_sis_customsizes' );
	} // register_settings()
	
	/**
	 * Load plugin options
	 */ 
	public function load_settings() {
		$options = get_option('cryout_sis_customsizes');
		$options = wp_parse_args( (array)$options, array() );
		return $options;
	} // load_settings()
	
	/**
	 * Save plugin options
	 */ 
	public function save_settings() {
		if ( isset( $_POST['settings_submit'] ) && check_admin_referer( 'cryout_sis_customsizes', '_cryout_sis_nonce' ) ):
			if (! current_user_can( 'manage_options' ) ) wp_die( __('You do not have the necessary permissions to perform this action.', 'cryout-sis') );
			$saved_options = $_POST['cryout_sis_customsizes'];

			foreach ($saved_options as $option => $value):
				if (!empty($value)) $saved_options[$option] = array_map( 'esc_attr', $value );;
			endforeach;
			
			update_option( 'cryout_sis_customsizes', $saved_options );
			wp_redirect( admin_url( $this->page_url . '&updated=true') );
		endif;
	} // save_settings()
	
	/**
	 * Register the plugin's menu page.
	 */
	public function register_menu_page() {
		$this->the_page = add_submenu_page(
			'options-general.php', 
			__('Serious Image Sizes', 'cryout-sis'), 
			__('Serious Image Sizes', 'cryout-sis'), 
			'manage_options', 
			'cryout-seriousimagesettings', 
			array( &$this, 'main_page' ) 
		);
	} // register_menu_page()

	/**
	 * Plugin Page
	 */
	public function main_page() {
	?>
	<div class="wrap" id="cryout-plugin-page">
		<h2><?php _e( 'Serious Image Sizes', 'cryout-sis' ) ?></h2>
					
		<div id="poststuff">
		<div id="post-body" class="metabox-holder columns-2">
			<div id="post-body-content">
			
				<div class="postbox">
					<div class="inner">
						<h3></h3>
						<p><?php _e( 'The existing values are presented as placeholders on the input fields. To override specific values, simply enter the new value in the field and save settings.', 'cryout-sis' ) ?></p>
						
						<form method="post" action="">
							<?php settings_fields( 'cryout_sis_group' ); ?>

						<div style="overflow-x:auto;">
						<table id="cryout-imagesizes" class="cryout-imagesizes">
						<thead>
							<tr><th class="size-name"><?php _e( 'Size name', 'cryout-sis' ) ?></th><th><?php _e( 'Width', 'cryout-sis' ) ?></th><th><?php _e( 'Height', 'cryout-sis' ) ?></th><th><?php _e( 'Crop Mode', 'cryout-sis' ) ?></th></tr>
							<tr><th colspan="4"><hr></th></tr>
						</thead>
						<tbody>
							<?php echo $this->render_sizes_table() ?>
						</tbody>
						<tfoot>
							<tr><td colspan="4"><hr><p>Parameter values are those supported by WordPress' <a href="https://developer.wordpress.org/reference/functions/add_image_size/" target="_blank">add_image_size()</a> function. <?php if ($this->known_theme) { ?>Values highlighted in <span style="background-color: rgba(0,128,0,0.1);">green</span> are image sizes registered by a Cryout theme.<?php } ?></p>
							<p>In the crop mode column enter <tt>0</tt> for no crop, <tt>1</tt> for auto crop (same as <tt>center center</tt>) or the two position values separated by space (eg. <tt>left top</tt>).</p>
							<p><strong>Remember to regenerate thumbnails after changing any image size values.</strong> Values highlighted in <span style="color: #800000; border-left: 4px solid #800000;"> red</span> are customized. Hover them to see original values.</p>
							<p></p>
							<?php wp_nonce_field( 'cryout_sis_customsizes', '_cryout_sis_nonce' ); ?>
							<?php submit_button( __('Save Custom Sizes','cryout-sis'), 'button-primary', 'settings_submit' ); ?>
							</td></tr>
						</tfoot>
						</table>
						</div><!--overflow-->
					
						</form>
					</div> 
				</div>			
						
			</div> <!-- post-body-content-->

			<div id="postbox-container-1" class="postbox-container">

						<div class="meta-box-sortables">

							<div class="postbox">
								<h3 style="text-align: center;" class="hndle">
									<span><strong><?php _e( 'Serious Image Sizes', 'cryout-sis') ?></strong></span>
								</h3>

								<div class="inside">
									<div style="text-align: center; margin: auto">
										<strong><?php printf( __('version: %s','cryout-sis'), $this->version ); ?></strong><br>
										<?php _e('by','cryout-sis') ?> Cryout Creations<br>
										<a target="_blank" href="http://www.cryoutcreations.eu/wordpress-plugins/serious-image-sizes/">www.cryoutcreations.eu</a>
									</div>
								</div>
							</div>

							<div class="postbox">
								<h3 style="text-align: center;" class="hndle">
									<span><?php _e('Support','cryout-sis') ?></span>
								</h3><div class="inside">
									<div style="text-align: center; margin: auto">
										<?php printf ( '%1$s <a target="_blank" href="http://www.cryoutcreations.eu/forum">%2$s</a>.',
											__('For support questions please use', 'cryout-sis'),
											__('our forum', 'cryout-sis')
											);
										?>
									</div>
								</div>
							</div>
							
						</div>
			</div> <!-- postbox-container -->

		</div> <!-- post-body -->
		<br class="clear">
		</div> <!-- poststuff -->

	</div><!--end wrap-->
	
	<?php
	} // main_page()

	/**
	 * Action Links
	 */
	public function actions_links( $links ) {
		array_unshift( $links, '<a href="' . $this->page_url . '">' . __( 'Configure', 'cryout-sis' ) . '</a>' );
		return $links;
	} // actions_links()

	/**
	 * Meta Links
	 */
	public function meta_links( $links, $file ) {
		// Check plugin
		if ( $file === plugin_basename( __FILE__ ) ) {
			unset( $links[2] );
			$links[] = '<a href="http://www.cryoutcreations.eu/cryout-imagesizes" target="_blank">' . __( 'Plugin homepage', 'cryout-serious-slider' ) . '</a>';
			$links[] = '<a href="https://www.cryoutcreations.eu/forums/f/wordpress/plugins/" target="_blank">' . __( 'Support forum', 'cryout-serious-slider' ) . '</a>';
		}
		return $links;
	} // meta_links()

	/**
	 * Main data manipulation block
	 */
	function parse_image_sizes() { 
		
		global $_wp_additional_image_sizes;

		$sizes = array();
		$get_intermediate_image_sizes = get_intermediate_image_sizes();

		// Create the full array with sizes and crop info
		foreach( $get_intermediate_image_sizes as $_size ) {

				if ( in_array( $_size, array( 'thumbnail', 'medium', 'large' ) ) ) {

						$sizes[ $_size ]['width'] = get_option( $_size . '_size_w' );
						$sizes[ $_size ]['height'] = get_option( $_size . '_size_h' );
						$sizes[ $_size ]['crop'] = (bool) get_option( $_size . '_crop' );
						
				} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {

						$sizes[ $_size ] = array( 
								'width' => $_wp_additional_image_sizes[ $_size ]['width'],
								'height' => $_wp_additional_image_sizes[ $_size ]['height'],
								'crop' =>  $_wp_additional_image_sizes[ $_size ]['crop']
						);
						
				}
		}

		$this->sizes = $sizes;
		
		// override customized values
		foreach ($sizes as $name => $values) {
			if (!empty( $this->options[$name] ) && is_array( $this->options[$name] ) ) {
				$replace = FALSE;
				$new_width = $values['width'];
				$new_height = $values['height'];
				$new_crop = $values['crop'];
				if (!empty( $this->options[$name]['width'] ) ) {
					$replace = TRUE;
					$new_width = intval( $this->options[$name]['width'] );
				}
				if (!empty( $this->options[$name]['height'] ) ) {
					$replace = TRUE;
					$new_height = intval( $this->options[$name]['height'] );
				}
				if ( $this->options[$name]['crop'] != '' ) {
					$replace = TRUE;
					if ( is_numeric( $this->options[$name]['crop'] ) ) $new_crop = boolval( $this->options[$name]['crop'] );
						else $new_crop = explode( " ", trim( $this->options[$name]['crop'] ) );
				}
				
				if ($replace) {
					remove_image_size( $name );
					add_image_size( $name, $new_width, $new_height, $new_crop );
				}
			}
		}

	} // parse_image_sizes()
	
	/**
	 * Sizes table wrapper 
	 */
	function render_sizes_table() {
		
		ob_start(); 
		
		foreach ($this->sizes as $name => $val) {
			$crop = (is_array($val['crop']) ? implode(' ',$val['crop']) : (($val['crop'] === true)?1:0) );
			
			$this->table_row( array( 'name'=>$name, 'width'=>$val['width'], 'height'=>$val['height'], 'crop'=>$crop ) );
		}
	
		return ob_get_clean();
	} // render_sizes_table()
	
	/**
	 * Helper function to build the form and display fields for each image size 
	 */
	function table_row( $args=array() ){
		
		$class = ''; $readonly = false;
		foreach ($this->known_themes as $theme) {
			if ( strpos( $args['name'], $theme ) !== false ) { 
				$class = ' known';
				$this->known_theme = TRUE;
			}			
		}
		
		if ( in_array( $args['name'], array( 'thumbnail', 'medium', 'large' ) ) ) $readonly = true;
		
		if ( isset( $this->options[$args['name']] ) && is_array( $this->options[$args['name']] ) ) {
			$width = $this->options[$args['name']]['width'];
			$height = $this->options[$args['name']]['height'];
			$crop = $this->options[$args['name']]['crop'];
			// todo: empty values
		}
		
		$name = trim($args['name']);
		$orig_width = intval($args['width']);
		$orig_height = intval($args['height']);
		$orig_crop = trim($args['crop']);
		
		?><tr class="row<?php echo $class ?>">
				<td class="size-name"><?php echo $name; ?></td>
				<?php foreach ( array( 'width', 'height', 'crop' ) as $id ) { ?>
				<td class="size-width <?php echo $this->is_customized( ${"orig_$id"}, ${$id} ) ?>">
					<input type="text" name="cryout_sis_customsizes[<?php echo $name; ?>][<?php echo $id ?>]" placeholder="<?php echo ${"orig_$id"} ?>" value="<?php echo ${$id} ?>" title="<?php echo $this->is_customized_title( ${"orig_$id"}, ${$id} ) ?>" <?php echo ($readonly?'readonly="readonly"':'') ?>>
				</td>
				<?php } ?>
		</tr><?php
		
	} // table_row()
	
	function is_customized( $original, $value = '' ) {
		if ( ( $value != '' ) && ( $original != $value ) ) return "customized";			
		return '';
	} // is_customized()
	
	function is_customized_title( $original, $value = '' ) {
		if ( ( $value != '' ) && ( $original != $value ) ) return sprintf( "Original value = %s", $original );	
		return '';
	} // is_customized_title()

	
	/**
	 * Load CSS for the plugin.
	 */
	function styling( $hook ) {
		if ( ($hook != $this->the_page) || ! current_user_can( 'manage_options' ) ) { return; }

		$plugin_url = trailingslashit( plugin_dir_url( __FILE__ ) );

		wp_enqueue_style( 'cryout-sis', $plugin_url . 'resources/style.css', 'screen', $this->version );
	} // styling()

	
	/**
	 * Load JavaScript for the plugin.
	 */
	function scripting( $hook ) {
		if ( ($hook != $this->the_page) || ! current_user_can( 'manage_options' ) ) { return; }

		$plugin_url = trailingslashit( plugin_dir_url( __FILE__ ) );

		wp_enqueue_script( 'cryout-sis', $plugin_url . 'resources/code.js', array( 'jquery' ), $this->version );
	} // scripting()

} // class Cryout_SeriousImageSizes

function cryout_seriousimagesizes_init(){
	global $cryout_seriousimagesizes;
	$cryout_seriousimagesizes = new Cryout_SeriousImageSizes;
}
add_action( 'init', 'cryout_seriousimagesizes_init' );

// FIN
